from flask import Blueprint, render_template, redirect, url_for, flash, request, jsonify
from flask_login import login_user, current_user, logout_user, login_required
from werkzeug.security import generate_password_hash, check_password_hash
from flask_wtf.csrf import generate_csrf

from app import db
from app.forms import RegistrationForm, LoginForm, TaskForm, TaskListForm
from app.models import User, Task, TaskList

# ✅ Define the blueprint
main = Blueprint('main', __name__)

@main.app_context_processor
def inject_csrf_token():
    return dict(csrf_token=generate_csrf)

@main.route('/')
def index():
    return render_template('index.html')

@main.route('/register', methods=['GET', 'POST'])
def register():
    if current_user.is_authenticated:
        return redirect(url_for('main.dashboard'))
    form = RegistrationForm()
    if form.validate_on_submit():
        hashed_pw = generate_password_hash(form.password.data)
        user = User(username=form.username.data, password=hashed_pw)
        db.session.add(user)
        db.session.commit()
        flash('Account created!', 'success')
        return redirect(url_for('main.login'))
    return render_template('register.html', form=form)

@main.route('/login', methods=['GET', 'POST'])
def login():
    if current_user.is_authenticated:
        return redirect(url_for('main.dashboard'))
    form = LoginForm()
    if form.validate_on_submit():
        user = User.query.filter_by(username=form.username.data).first()
        if user and check_password_hash(user.password, form.password.data):
            login_user(user)
            return redirect(url_for('main.dashboard'))
        flash('Login failed. Check your credentials.', 'danger')
    return render_template('login.html', form=form)

@main.route('/logout')
def logout():
    logout_user()
    return redirect(url_for('main.index'))

@main.route('/dashboard')
@login_required
def dashboard():
    task_lists = TaskList.query.filter_by(user_id=current_user.id).all()
    return render_template('dashboard.html', task_lists=task_lists)

@main.route('/create_list', methods=['GET', 'POST'])
@login_required
def create_list():
    form = TaskListForm()
    if form.validate_on_submit():
        new_list = TaskList(name=form.name.data, user_id=current_user.id)
        db.session.add(new_list)
        db.session.commit()
        flash('Task list created!', 'success')
        return redirect(url_for('main.dashboard'))
    return render_template('create_list.html', form=form)

@main.route('/add_task', methods=['GET', 'POST'])
@login_required
def add_task():
    form = TaskForm()
    form.list_id.choices = [(l.id, l.name) for l in TaskList.query.filter_by(user_id=current_user.id).all()]
    if form.validate_on_submit():
        task = Task(
            content=form.content.data,
            list_id=form.list_id.data,
            user_id=current_user.id
        )
        db.session.add(task)
        db.session.commit()
        flash('Task added!', 'success')
        return redirect(url_for('main.dashboard'))
    return render_template('add_task.html', form=form)

@main.route('/complete/<int:task_id>')
@login_required
def complete(task_id):
    task = Task.query.get_or_404(task_id)
    if task.task_list.user_id != current_user.id:
        return redirect(url_for('main.dashboard'))
    task.completed = not task.completed
    db.session.commit()
    return redirect(url_for('main.dashboard'))

@main.route('/delete/<int:task_id>')
@login_required
def delete(task_id):
    task = Task.query.get_or_404(task_id)
    if task.task_list.user_id != current_user.id:
        return redirect(url_for('main.dashboard'))
    db.session.delete(task)
    db.session.commit()
    return redirect(url_for('main.dashboard'))

@main.route('/move_task', methods=['POST'])
@login_required
def move_task():
    data = request.get_json()
    task_id = data.get('task_id')
    target_list_id = data.get('target_list_id')

    task = Task.query.get_or_404(task_id)
    target_list = TaskList.query.get_or_404(target_list_id)

    if task.task_list.user_id != current_user.id or target_list.user_id != current_user.id:
        return jsonify({'error': 'Unauthorized'}), 403

    task.list_id = target_list_id
    db.session.commit()
    return jsonify({'success': True})
